<?php

namespace App\Http\Controllers;

use App\Models\CandidateProfile;
use App\Models\JobApplication;
use App\Models\UserCourse;
use App\Models\Interview;
use App\Models\EducationBackground;
use App\Models\EmploymentHistory;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;

class CandidateProfileController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $request->validate([
            'skills' => 'string|nullable',
            'location' => 'string|nullable',
            'min_experience' => 'integer|min:0|nullable',
            'certifications' => 'string|nullable',
        ]);

        $query = CandidateProfile::with(['user', 'rate', 'availability', 'availableFor']);

        if ($request->has('skills')) {
            $skills = array_map('trim', explode(',', $request->input('skills')));
            $query->whereHas('user.skills', function ($q) use ($skills) {
                $q->whereIn('Skill_name', $skills);
            });
        }

        if ($request->has('location')) {
            $query->where('Location', 'LIKE', '%' . $request->input('location') . '%');
        }

        if ($request->has('min_experience')) {
            // Convert min_experience to a string with potential + prefix for varchar comparison
            $minExp = (string)$request->input('min_experience');
            $minExpFormatted = '+' . $minExp . ' years of experience'; // Match database format
            $query->where('Experience', '>=', $minExpFormatted);
        }

        if ($request->has('certifications')) {
            $certifications = array_map('trim', explode(',', $request->input('certifications')));
            $query->where(function ($q) use ($certifications) {
                foreach ($certifications as $cert) {
                    $q->whereJsonContains('Certifications', $cert);
                }
            });
        }

        try {
            $candidates = $query->paginate(10);
            // Preserve the original Experience value and check length
            foreach ($candidates->items() as $candidate) {
                if ($candidate->Experience && strlen($candidate->Experience) > 255) {
                    \Illuminate\Support\Facades\Log::warning('Experience value for candidate ID ' . $candidate->Profile_id . ' exceeds varchar(255) limit: ' . $candidate->Experience);
                    $candidate->Experience = substr($candidate->Experience, 0, 255); // Truncate if needed
                }
            }
            return response()->json([
                'status' => 'success',
                'count' => $candidates->total(),
                'data' => $candidates->items(),
                'pagination' => [
                    'current_page' => $candidates->currentPage(),
                    'start_page' => 1,
                    'end_page' => $candidates->lastPage(),
                    'per_page' => $candidates->perPage(),
                    'total' => $candidates->total(),
                ],
            ], 200);
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Failed to search candidates: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Server error: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function show($id): JsonResponse
    {
        try {
            $candidate = CandidateProfile::with(['user', 'rate', 'availability', 'availableFor'])->findOrFail($id);

            // Preserve the original Experience value
            if ($candidate->Experience && strlen($candidate->Experience) > 255) {
                \Illuminate\Support\Facades\Log::warning('Experience value for candidate ID ' . $id . ' exceeds varchar(255) limit: ' . $candidate->Experience);
                $candidate->Experience = substr($candidate->Experience, 0, 255); // Truncate if needed
            }

            return response()->json([
                'status' => 'success',
                'data' => $candidate,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Candidate not found: ' . $e->getMessage(),
            ], 404);
        }
    }

    public function search(Request $request): JsonResponse
    {
        $request->validate([
            'search' => 'string|nullable',
            'min_experience' => 'integer|min:0|nullable',
            'max_experience' => 'integer|min:0|nullable|gte:min_experience',
            'page' => 'integer|min:1|nullable',
            'per_page' => 'integer|min:1|max:10|nullable',
        ]);

        try {
            $query = CandidateProfile::with(['user', 'user.skills']);

            \Illuminate\Support\Facades\Log::info('Search parameters', $request->all());

            if ($request->has('search') && !empty($request->input('search'))) {
                $search = trim($request->input('search'));
                \Illuminate\Support\Facades\Log::info('General search filter applied', ['search' => $search]);
                $query->where(function ($q) use ($search) {
                    $q->whereRaw('LOWER(First_name) LIKE ?', ['%' . strtolower($search) . '%'])
                        ->orWhereRaw('LOWER(Last_name) LIKE ?', ['%' . strtolower($search) . '%'])
                        ->orWhereRaw('LOWER(CONCAT(First_name, \' \', Last_name)) LIKE ?', ['%' . strtolower($search) . '%'])
                        ->orWhereRaw('LOWER(Location) LIKE ?', ['%' . strtolower($search) . '%'])
                        ->orWhereHas('user.skills', function ($subQuery) use ($search) {
                            $subQuery->whereRaw('LOWER(Skill_name) LIKE ?', ['%' . strtolower($search) . '%']);
                        });
                });
            }

            if ($request->has('min_experience')) {
                \Illuminate\Support\Facades\Log::info('Min experience filter applied', ['min_experience' => $request->input('min_experience')]);
                $query->where('Experience', '>=', $request->input('min_experience'));
            }
            if ($request->has('max_experience')) {
                \Illuminate\Support\Facades\Log::info('Max experience filter applied', ['max_experience' => $request->input('max_experience')]);
                $query->where('Experience', '<=', $request->input('max_experience'));
            }

            \Illuminate\Support\Facades\Log::info('Generated SQL query', [
                'sql' => $query->toSql(),
                'bindings' => $query->getBindings()
            ]);

            $perPage = $request->input('per_page', 10);
            $candidates = $query->paginate($perPage);

            return response()->json([
                'status' => 'success',
                'count' => $candidates->total(),
                'data' => $candidates->items(),
                'pagination' => [
                    'current_page' => $candidates->currentPage(),
                    'start_page' => 1,
                    'end_page' => $candidates->lastPage(),
                    'per_page' => $candidates->perPage(),
                    'total' => $candidates->total(),
                ],
            ], 200);
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Failed to search candidates: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Server error: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'required|exists:users,User_id',
            'First_name' => 'nullable|string|max:100',
            'Last_name' => 'nullable|string|max:100',
            'Resume_url' => 'nullable|file|mimes:pdf,doc,docx|max:10240',
            'Certifications' => 'nullable|array',
            'Certifications.*' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:10240',
            'Location' => 'nullable|string|max:255',
            'Experience' => 'nullable|string|max:100',
            'bio' => 'nullable|string',
            'age' => 'nullable|integer|min:0',
            'photo' => 'nullable|file|mimes:jpg,jpeg,png|max:10240',
            'traits' => 'nullable|array',
            'references' => 'nullable|array',
            'references.*' => 'nullable|string',
            'child_age_prefs' => 'nullable|array',
            'verified_by_admin' => 'nullable|boolean',
            'is_active' => 'nullable|boolean',
        ], [
            'Resume_url.file' => 'The resume must be a valid file.',
            'Resume_url.mimes' => 'The resume must be a PDF, DOC, or DOCX file.',
            'Resume_url.max' => 'The resume must not exceed 10MB.',
            'Certifications.*.file' => 'Each certification must be a valid file.',
            'Certifications.*.mimes' => 'Each certification must be a JPG, JPEG, PNG, or PDF file.',
            'Certifications.*.max' => 'Each certification must not exceed 10MB.',
            'photo.file' => 'The photo must be a valid file.',
            'photo.mimes' => 'The photo must be a JPG, JPEG, or PNG file.',
            'photo.max' => 'The photo must not exceed 10MB.',
            'references.*.string' => 'Each reference must be a string.',
            'verified_by_admin.boolean' => 'The verified_by_admin field must be a boolean.',
            'is_active.boolean' => 'The is_active field must be a boolean.',
            'age.integer' => 'The age must be an integer.',
            'age.min' => 'The age must be at least 0.',
            'traits.array' => 'The traits must be an array.',
            'child_age_prefs.array' => 'The child_age_prefs must be an array.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'User_id',
            'First_name',
            'Last_name',
            'Resume_url',
            'Certifications',
            'Location',
            'Experience',
            'bio',
            'age',
            'traits',
            'references',
            'child_age_prefs',
            'verified_by_admin',
            'is_active',
        ]);

        // Handle Resume_url
        if ($request->hasFile('Resume_url') && $request->file('Resume_url')->isValid()) {
            $resume = $request->file('Resume_url');
            $resumeName = time() . '_resume_' . $resume->getClientOriginalName();
            $resumeDir = public_path('storage/resumes');

            if (!File::exists($resumeDir)) {
                File::makeDirectory($resumeDir, 0755, true);
                Log::info('Created directory: ' . $resumeDir);
            }

            $resume->move($resumeDir, $resumeName);
            $resumePath = public_path('storage/resumes/' . $resumeName);

            if (!File::exists($resumePath)) {
                Log::error('Failed to store resume at: ' . $resumePath);
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to store resume file',
                ], 500);
            }

            Log::info('Resume stored at: ' . $resumePath);
            $data['Resume_url'] = url('storage/resumes/' . $resumeName);
            Log::info('Generated Resume_url: ' . $data['Resume_url']);
        }

        // Handle Certifications
        if ($request->hasFile('Certifications')) {
            $certificationUrls = [];
            foreach ($request->file('Certifications') as $certification) {
                if ($certification->isValid()) {
                    $certName = time() . '_cert_' . $certification->getClientOriginalName();
                    $certDir = public_path('storage/certifications');

                    if (!File::exists($certDir)) {
                        File::makeDirectory($certDir, 0755, true);
                        Log::info('Created directory: ' . $certDir);
                    }

                    $certification->move($certDir, $certName);
                    $certPath = public_path('storage/certifications/' . $certName);

                    if (!File::exists($certPath)) {
                        Log::error('Failed to store certification at: ' . $certPath);
                        continue;
                    }

                    Log::info('Certification stored at: ' . $certPath);
                    $certUrl = url('storage/certifications/' . $certName);
                    $certificationUrls[] = $certUrl;
                    Log::info('Generated Cert_url: ' . $certUrl);
                }
            }
            if (!empty($certificationUrls)) {
                $data['Certifications'] = $certificationUrls;
            }
        }

        // Handle Photo (single)
        if ($request->hasFile('photo')) {
            $photo = $request->file('photo');
            if ($photo->isValid()) {
                $photoName = time() . '_photo_' . $photo->getClientOriginalName();
                $photoDir = public_path('storage/photos');

                if (!File::exists($photoDir)) {
                    File::makeDirectory($photoDir, 0755, true);
                    Log::info('Created directory: ' . $photoDir);
                }

                $photo->move($photoDir, $photoName);
                $photoPath = public_path('storage/photos/' . $photoName);

                if (!File::exists($photoPath)) {
                    Log::error('Failed to store photo at: ' . $photoPath);
                    return response()->json([
                        'status' => 'error',
                        'message' => 'Failed to store photo file',
                    ], 500);
                }

                Log::info('Photo stored at: ' . $photoPath);
                $data['photo'] = url('storage/photos/' . $photoName);
                Log::info('Generated Photo_url: ' . $data['photo']);
            }
        }

        if ($request->has('references')) {
            $data['references'] = $request->input('references');
        }

        if ($request->has('traits')) {
            $data['traits'] = $request->input('traits');
        }

        if ($request->has('child_age_prefs')) {
            $data['child_age_prefs'] = $request->input('child_age_prefs');
        }

        // Only superadmins can set verified_by_admin
        if ($request->has('verified_by_admin') && (!auth()->check() || !auth()->user()->superadmin)) {
            unset($data['verified_by_admin']);
        }

        $profile = CandidateProfile::create($data);

        return response()->json([
            'status' => 'success',
            'data' => $profile,
            'message' => 'Candidate profile created successfully',
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $profile = CandidateProfile::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'First_name' => 'nullable|string|max:100',
            'Last_name' => 'nullable|string|max:100',
            'Resume_url' => 'nullable|string', // Changed to string to allow base64 or URL
            'Certifications' => 'nullable|array',
            'Certifications.*' => 'nullable|string',
            'Location' => 'nullable|string|max:255',
            'Experience' => 'nullable|string|max:100',
            'bio' => 'nullable|string',
            'age' => 'nullable|integer|min:0',
            'photo' => 'nullable|string',
            'traits' => 'nullable|array',
            'references' => 'nullable|array',
            'references.*' => 'nullable|string',
            'child_age_prefs' => 'nullable|array',
            'verified_by_admin' => 'nullable|boolean',
            'is_active' => 'nullable|boolean',
        ], [
            'Resume_url.string' => 'The resume must be a valid base64 string or URL.',
            'Certifications.*.string' => 'Each certification must be a string.',
            'photo.string' => 'The photo must be a string.',
            'references.*.string' => 'Each reference must be a string.',
            'verified_by_admin.boolean' => 'The verified_by_admin field must be a boolean.',
            'is_active.boolean' => 'The is_active field must be a boolean.',
            'age.integer' => 'The age must be an integer.',
            'age.min' => 'The age must be at least 0.',
            'traits.array' => 'The traits must be an array.',
            'child_age_prefs.array' => 'The child_age_prefs must be an array.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'First_name',
            'Last_name',
            'Resume_url',
            'Certifications',
            'Location',
            'Experience',
            'bio',
            'age',
            'traits',
            'references',
            'child_age_prefs',
            'verified_by_admin',
            'is_active',
        ]);

        // Helper function to validate base64
        if (!function_exists('isValidBase64')) {
            function isValidBase64($string)
            {
                return (bool) preg_match('/^[A-Za-z0-9+\/=]+$/', $string) && base64_decode($string, true) !== false;
            }
        }

        // Helper function to get mime type from base64
        if (!function_exists('getMimeTypeFromBase64')) {
            function getMimeTypeFromBase64($base64String)
            {
                if (preg_match('/^data:(image|application)\/([a-z]+);base64,/', $base64String, $matches)) {
                    return $matches[1] . '/' . $matches[2];
                }
                return 'application/octet-stream';
            }
        }

        // Handle Resume_url (base64, URL, or file)
        if ($request->has('Resume_url')) {
            $resumeInput = $request->input('Resume_url');

            if (empty($resumeInput)) {
                // Clear resume if empty input
                if ($profile->Resume_url) {
                    $oldResumePath = str_replace(url('/storage/'), 'public/', $profile->Resume_url);
                    if (Storage::exists($oldResumePath)) {
                        Storage::delete($oldResumePath);
                    }
                }
                $data['Resume_url'] = null;
            } else {
                $base64String = null;
                $mimeType = 'application/pdf';
                $isFile = $request->hasFile('Resume_url') && $request->file('Resume_url')->isValid();

                if (!$isFile) {
                    // Check for base64
                    if (preg_match('/^data:application\/(pdf|msword|doc|docx);base64,/', $resumeInput)) {
                        $base64String = preg_replace('/^data:application\/(pdf|msword|doc|docx);base64,/', '', $resumeInput);
                        $mimeType = getMimeTypeFromBase64($resumeInput);
                    } elseif (isValidBase64($resumeInput)) {
                        $base64String = $resumeInput;
                    }
                }

                if ($isFile || $base64String) {
                    // Validate file or base64
                    if ($isFile) {
                        $resume = $request->file('Resume_url');
                        $extension = strtolower($resume->getClientOriginalExtension());
                        $fileSize = $resume->getSize();
                        $resumeName = time() . '_resume_' . $resume->getClientOriginalName();
                    } else {
                        $resumeData = base64_decode($base64String, true);
                        if ($resumeData === false) {
                            return response()->json([
                                'status' => 'error',
                                'message' => 'Invalid base64 for resume',
                            ], 422);
                        }
                        $fileSize = strlen($resumeData);
                        $extension = [
                            'application/pdf' => 'pdf',
                            'application/msword' => 'doc',
                            'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => 'docx',
                        ][$mimeType] ?? 'pdf';
                        $resumeName = time() . '_resume.' . $extension;
                    }

                    // Validate mime type and size
                    $allowedMimes = ['pdf', 'doc', 'docx'];
                    if (!in_array($extension, $allowedMimes)) {
                        return response()->json([
                            'status' => 'error',
                            'errors' => ['Resume_url' => ['The resume must be a PDF, DOC, or DOCX file.']],
                        ], 422);
                    }
                    if ($fileSize > 10240 * 1024) {
                        return response()->json([
                            'status' => 'error',
                            'errors' => ['Resume_url' => ['The resume must not exceed 10MB.']],
                        ], 422);
                    }

                    // Delete old resume if exists
                    if ($profile->Resume_url) {
                        $oldResumePath = str_replace(url('/storage/'), 'public/', $profile->Resume_url);
                        if (Storage::exists($oldResumePath)) {
                            Storage::delete($oldResumePath);
                        }
                    }

                    // Store new resume
                    $resumeDir = public_path('storage/resumes');
                    if (!File::exists($resumeDir)) {
                        File::makeDirectory($resumeDir, 0755, true);
                    }

                    if ($isFile) {
                        $resume->move($resumeDir, $resumeName);
                    } else {
                        file_put_contents($resumeDir . '/' . $resumeName, $resumeData);
                    }

                    $resumePath = $resumeDir . '/' . $resumeName;
                    if (!File::exists($resumePath)) {
                        Log::error('Failed to store resume at: ' . $resumePath);
                        return response()->json([
                            'status' => 'error',
                            'message' => 'Failed to store resume file',
                        ], 500);
                    }

                    $data['Resume_url'] = url('storage/resumes/' . $resumeName);
                } elseif (filter_var($resumeInput, FILTER_VALIDATE_URL) && strpos($resumeInput, url('storage/resumes/')) === 0) {
                    // Retain existing URL if valid
                    $data['Resume_url'] = $resumeInput;
                } else {
                    return response()->json([
                        'status' => 'error',
                        'errors' => ['Resume_url' => ['Invalid resume input. Must be a valid base64 string, file, or storage URL.']],
                    ], 422);
                }
            }
        }

        // Handle Certifications (unchanged from original)
        if ($request->has('Certifications')) {
            $newCertificationUrls = $profile->Certifications ?? [];
            $inputCertifications = $request->input('Certifications', []);

            foreach ($inputCertifications as $index => $certInput) {
                if (empty($certInput)) {
                    continue;
                }

                $base64String = null;
                $mimeType = 'image/jpeg';
                if (preg_match('/^data:(image|application)\/(jpeg|jpg|png|pdf);base64,/', $certInput)) {
                    $base64String = preg_replace('/^data:(image|application)\/(jpeg|jpg|png|pdf);base64,/', '', $certInput);
                    $mimeType = getMimeTypeFromBase64($certInput);
                } elseif (isValidBase64($certInput)) {
                    $base64String = $certInput;
                }

                if ($base64String) {
                    if (!isValidBase64($base64String)) {
                        continue;
                    }
                    $certData = base64_decode($base64String, true);
                    if ($certData === false) {
                        continue;
                    }
                    if (strlen($certData) > 10240 * 1024) {
                        continue;
                    }

                    if (isset($newCertificationUrls[$index])) {
                        $oldCertPath = str_replace(url('/storage/'), 'public/', $newCertificationUrls[$index]);
                        if (Storage::exists($oldCertPath)) {
                            Storage::delete($oldCertPath);
                        }
                    }

                    $certExtension = [
                        'image/jpeg' => 'jpg',
                        'image/jpg' => 'jpg',
                        'image/png' => 'png',
                        'application/pdf' => 'pdf',
                    ][$mimeType] ?? 'jpg';
                    $certName = time() . '_cert_' . $index . '.' . $certExtension;
                    $certDir = public_path('storage/certifications');

                    if (!File::exists($certDir)) {
                        File::makeDirectory($certDir, 0755, true);
                    }

                    $certPath = $certDir . '/' . $certName;
                    file_put_contents($certPath, $certData);

                    if (!File::exists($certPath)) {
                        Log::error('Failed to store certification at: ' . $certPath);
                        continue;
                    }

                    $newCertificationUrls[$index] = url('storage/certifications/' . $certName);
                } elseif (filter_var($certInput, FILTER_VALIDATE_URL) && strpos($certInput, url('storage/certifications/')) === 0) {
                    $newCertificationUrls[$index] = $certInput;
                } else {
                    unset($newCertificationUrls[$index]);
                }
            }

            $data['Certifications'] = !empty($newCertificationUrls) ? array_values($newCertificationUrls) : null;
        }

        // Handle Photo (unchanged from original)
        if ($request->has('photo')) {
            $photoInput = $request->input('photo');

            if (empty($photoInput)) {
                if ($profile->photo) {
                    $oldPhotoPath = str_replace(url('/storage/'), 'public/', $profile->photo);
                    if (Storage::exists($oldPhotoPath)) {
                        Storage::delete($oldPhotoPath);
                    }
                }
                $data['photo'] = null;
            } else {
                $base64String = null;
                $mimeType = 'image/jpeg';
                if (preg_match('/^data:image\/(jpeg|jpg|png);base64,/', $photoInput)) {
                    $base64String = preg_replace('/^data:image\/(jpeg|jpg|png);base64,/', '', $photoInput);
                    $mimeType = getMimeTypeFromBase64($photoInput);
                } elseif (isValidBase64($photoInput)) {
                    $base64String = $photoInput;
                }

                if ($base64String) {
                    if (!isValidBase64($base64String)) {
                        return response()->json([
                            'status' => 'error',
                            'message' => 'Invalid base64 for photo',
                        ], 422);
                    }
                    $photoData = base64_decode($base64String, true);
                    if ($photoData === false) {
                        return response()->json([
                            'status' => 'error',
                            'message' => 'Failed to decode base64 for photo',
                        ], 422);
                    }
                    if (strlen($photoData) > 10240 * 1024) {
                        return response()->json([
                            'status' => 'error',
                            'message' => 'Photo size exceeds 10MB',
                        ], 422);
                    }

                    if ($profile->photo) {
                        $oldPhotoPath = str_replace(url('/storage/'), 'public/', $profile->photo);
                        if (Storage::exists($oldPhotoPath)) {
                            Storage::delete($oldPhotoPath);
                        }
                    }

                    $photoExtension = [
                        'image/jpeg' => 'jpg',
                        'image/jpg' => 'jpg',
                        'image/png' => 'png'
                    ][$mimeType] ?? 'jpg';
                    $photoName = time() . '_photo.' . $photoExtension;
                    $photoDir = public_path('storage/photos');

                    if (!File::exists($photoDir)) {
                        File::makeDirectory($photoDir, 0755, true);
                        Log::info('Created directory: ' . $photoDir);
                    }

                    $photoPath = $photoDir . '/' . $photoName;
                    file_put_contents($photoPath, $photoData);

                    if (!File::exists($photoPath)) {
                        Log::error('Failed to store photo at: ' . $photoPath);
                        return response()->json([
                            'status' => 'error',
                            'message' => 'Failed to store photo',
                        ], 500);
                    }

                    $data['photo'] = url('storage/photos/' . $photoName);
                } elseif (filter_var($photoInput, FILTER_VALIDATE_URL) && strpos($photoInput, url('storage/photos/')) === 0) {
                    $data['photo'] = $photoInput;
                } else {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'Invalid photo input',
                    ], 422);
                }
            }
        }

        if ($request->has('references')) {
            $data['references'] = $request->input('references');
        }

        if ($request->has('traits')) {
            $data['traits'] = $request->input('traits');
        }

        if ($request->has('child_age_prefs')) {
            $data['child_age_prefs'] = $request->input('child_age_prefs');
        }

        // Only superadmins can update verified_by_admin
        if ($request->has('verified_by_admin') && (!auth()->check() || !auth()->user()->superadmin)) {
            unset($data['verified_by_admin']);
        }

        $profile->update($data);

        return response()->json([
            'status' => 'success',
            'data' => $profile,
            'message' => 'Profile updated successfully',
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        try {
            $profile = CandidateProfile::findOrFail($id);

            // Delete associated files if they exist
            if ($profile->Resume_url) {
                $resumePath = str_replace(url('/storage/'), 'public/storage/', $profile->Resume_url);
                if (Storage::exists($resumePath)) {
                    Storage::delete($resumePath);
                }
            }

            if ($profile->Certifications) {
                foreach ($profile->Certifications as $certUrl) {
                    $certPath = str_replace(url('/storage/'), 'public/storage/', $certUrl);
                    if (Storage::exists($certPath)) {
                        Storage::delete($certPath);
                    }
                }
            }

            if ($profile->photo) {
                $photoPath = str_replace(url('/storage/'), 'public/storage/', $profile->photo);
                if (Storage::exists($photoPath)) {
                    Storage::delete($photoPath);
                }
            }

            $profile->delete();

            return response()->json([
                'status' => 'success',
                'message' => 'Candidate profile deleted successfully',
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to delete candidate profile: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function deleteResume(Request $request, $id): JsonResponse
    {
        $profile = CandidateProfile::findOrFail($id);

        // Delete resume file if it exists
        if ($profile->Resume_url) {
            $resumePath = str_replace(url('/storage/'), 'public/', $profile->Resume_url);
            if (Storage::exists($resumePath)) {
                Storage::delete($resumePath);
            }
            $profile->Resume_url = null;
            $profile->save();
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Resume deleted successfully',
            'data' => $profile,
        ], 200);
    }

    public function deleteCertification(Request $request, $id): JsonResponse
    {
        $profile = CandidateProfile::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'cert_url' => 'nullable|string|url',
            'cert_index' => 'nullable|integer|min:0',
        ], [
            'cert_url.string' => 'The cert_url must be a string.',
            'cert_url.url' => 'The cert_url must be a valid URL.',
            'cert_index.integer' => 'The cert_index must be an integer.',
            'cert_index.min' => 'The cert_index must be at least 0.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $certifications = $profile->Certifications ?? [];
        $certUrl = $request->input('cert_url');
        $certIndex = $request->input('cert_index');

        if ($certIndex !== null && isset($certifications[$certIndex])) {
            $certUrl = $certifications[$certIndex];
        } elseif ($certUrl && !in_array($certUrl, $certifications)) {
            return response()->json([
                'status' => 'error',
                'message' => 'Certification not found in profile. Provided URL: ' . $certUrl . ', Available URLs: ' . json_encode($certifications, JSON_UNESCAPED_SLASHES),
            ], 404);
        }

        if ($certUrl) {
            $certifications = array_filter($certifications, function ($url) use ($certUrl) {
                return $url !== $certUrl;
            });

            $certPath = str_replace(url('/storage/'), 'public/storage/', $certUrl);
            if (Storage::exists($certPath)) {
                Storage::delete($certPath);
            }

            $profile->update([
                'Certifications' => !empty($certifications) ? array_values($certifications) : null,
            ]);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Certification deleted successfully',
            'data' => $profile,
        ], 200);
    }

    public function deletePhoto(Request $request, $id): JsonResponse
    {
        $profile = CandidateProfile::findOrFail($id);

        if ($profile->photo) {
            $photoPath = str_replace(url('/storage/'), 'public/storage/', $profile->photo);
            if (Storage::exists($photoPath)) {
                Storage::delete($photoPath);
            }

            $profile->update([
                'photo' => null,
            ]);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Photo deleted successfully',
            'data' => $profile,
        ], 200);
    }

    private function getMimeTypeFromBase64($base64String)
    {
        $data = explode(';', $base64String)[0];
        return str_replace('data:', '', $data);
    }

    private function isValidBase64($string)
    {
        if (!preg_match('/^[A-Za-z0-9+\/=]+$/', $string)) {
            return false;
        }
        if (strlen($string) % 4 !== 0) {
            return false;
        }
        return true;
    }

    public function getStats($Candidate_id): JsonResponse
    {
        $profile = CandidateProfile::findOrFail($Candidate_id);
        $userId = $profile->User_id;

        $jobApplications = JobApplication::where('User_id', $userId)->count();

        $coursesTotal = UserCourse::where('User_id', $userId)->count();
        $coursesCompleted = UserCourse::where('User_id', $userId)->where('Status', 'completed')->count();
        $coursesPending = UserCourse::where('User_id', $userId)->where('Status', 'pending')->count();

        $applicationIds = JobApplication::where('User_id', $userId)->pluck('Application_id')->toArray();

        $interviews = Interview::whereIn('Application_id', $applicationIds)
            ->groupBy('Status')
            ->select('Status', DB::raw('count(*) as count'))
            ->pluck('count', 'Status')
            ->toArray();

        $interviewStats = [
            'scheduled' => $interviews['scheduled'] ?? 0,
            'completed' => $interviews['completed'] ?? 0,
            'cancelled' => $interviews['cancelled'] ?? 0,
            'mock' => $interviews['mock'] ?? 0,
        ];

        return response()->json([
            'status' => 'success',
            'data' => [
                'job_applications' => $jobApplications,
                'courses' => [
                    'total' => $coursesTotal,
                    'completed' => $coursesCompleted,
                    'pending' => $coursesPending,
                ],
                'interviews' => $interviewStats,
            ],
        ], 200);
    }

    public function getFullProfile($userId): JsonResponse
    {
        try {
            $user = User::find($userId);
            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'User not found for the provided User ID',
                ], 404);
            }

            $profile = $user->candidateProfile()->with(['user', 'user.skills'])->first();
            if (!$profile) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'No candidate profile found for the provided User ID',
                ], 404);
            }

            $education = EducationBackground::where('User_id', $userId)->get();
            $employment = EmploymentHistory::where('User_id', $userId)->get();
            $courses = UserCourse::where('User_id', $userId)->with('course')->get();

            return response()->json([
                'status' => 'success',
                'data' => [
                    'profile' => $profile,
                    'education_background' => $education,
                    'employment_history' => $employment,
                    'user_courses' => $courses,
                ],
                'message' => 'Full candidate profile retrieved successfully',
            ], 200);
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Failed to retrieve full profile: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Server error: ' . $e->getMessage(),
            ], 500);
        }
    }
}
